/* useralias.c -- Users aliases maintenance. Users are kept in a binary tree.
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,96  Lele Gaifax
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include    "tua.h"

typedef struct user_aliases
{
  char *User;
  char *Alias;
} user_aliases_t;

static BTREE Root = NULL;

static int
DEFUN (alias_compare, (user1, user2),
       CONST user_aliases_t * user1 AND CONST user_aliases_t * user2)
{
  int ret = strcmp (user1->User, user2->User);

  return ret;
}

static PTR
DEFUN (alias_new, (data),
       CONST PTR * data)
{
  user_aliases_t * sa;
  
  sa = (user_aliases_t *) xmalloc (sizeof (user_aliases_t));
  sa->User = savestring (((CONST user_aliases_t *) data)->User);
  sa->Alias = NULL;

  return (PTR) sa;
}
       
CONST char *
DEFUN (insert_user_alias, (user, alias),
       CONST char *user AND
       CONST char *alias)
{
  user_aliases_t * ua;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) alias_compare, (makenew_func_t) alias_new);

  ua = (user_aliases_t *) btree_insert (Root, (PTR)user);
  if (! ua->Alias)
    ua->Alias = savestring (alias);
  
  return ua->Alias;
}

CONST char *
DEFUN (get_possible_user_alias, (user),
       CONST char *user)
{
  user_aliases_t * ua, ua_key;

  if (Root == NULL)
    return user;

  ua_key.User = (char *) user;
  ua = (user_aliases_t *) btree_search (Root, &ua_key);
  
#if USE_GLOB_ALIASES
  if (ua == NULL)
    return get_possible_user_glob_alias (user);
#endif   

  if (ua == NULL)
    return user;
  else
    return ua->Alias;
}

#if USE_GLOB_ALIASES

#include "globalias.h"

static glob_aliases_queue_t *UserGlobQueue = QNULL;

void
DEFUN (insert_user_glob_alias, (user, alias),
       CONST char *user AND
       CONST char *alias)
{
  globalias_insert (&UserGlobQueue, user, alias);
}

CONST char *
DEFUN (get_possible_user_glob_alias, (user),
       CONST char *user)
{
  return globalias_search (UserGlobQueue, user);
}

#endif /* if USE_GLOB_ALIASES */

#if USE_TCL

int
DEFUN (tcl_insert_user_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, _("Wrong number of Parameters: %s needs an UserName and its Alias"), argv[0]);
      return TCL_ERROR;
    }
  insert_user_alias (argv[1], argv[2]);
  return TCL_OK;
}

#if USE_GLOB_ALIASES

int
DEFUN (tcl_insert_user_glob_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, _("Wrong number of Parameters: %s needs an UserName and its GlobAlias"), argv[0]);
      return TCL_ERROR;
    }
  insert_user_glob_alias (argv[1], argv[2]);
  return TCL_OK;
}

#endif /* if USE_GLOB_ALIASES */

#endif /* if USE_TCL */
