#include "reedsolomon.h"
#include <cstring>

namespace reedsolomon
{
    unsigned char ToDualBasis[256] = {0x00, 0x7b, 0xaf, 0xd4, 0x99, 0xe2, 0x36, 0x4d, 0xfa, 0x81, 0x55, 0x2e, 0x63, 0x18, 0xcc, 0xb7, 0x86, 0xfd, 0x29, 0x52, 0x1f,
                                      0x64, 0xb0, 0xcb, 0x7c, 0x07, 0xd3, 0xa8, 0xe5, 0x9e, 0x4a, 0x31, 0xec, 0x97, 0x43, 0x38, 0x75, 0x0e, 0xda, 0xa1, 0x16, 0x6d, 0xb9, 0xc2, 0x8f, 0xf4,
                                      0x20, 0x5b, 0x6a, 0x11, 0xc5, 0xbe, 0xf3, 0x88, 0x5c, 0x27, 0x90, 0xeb, 0x3f, 0x44, 0x09, 0x72, 0xa6, 0xdd, 0xef, 0x94, 0x40, 0x3b, 0x76, 0x0d, 0xd9,
                                      0xa2, 0x15, 0x6e, 0xba, 0xc1, 0x8c, 0xf7, 0x23, 0x58, 0x69, 0x12, 0xc6, 0xbd, 0xf0, 0x8b, 0x5f, 0x24, 0x93, 0xe8, 0x3c, 0x47, 0x0a, 0x71, 0xa5, 0xde,
                                      0x03, 0x78, 0xac, 0xd7, 0x9a, 0xe1, 0x35, 0x4e, 0xf9, 0x82, 0x56, 0x2d, 0x60, 0x1b, 0xcf, 0xb4, 0x85, 0xfe, 0x2a, 0x51, 0x1c, 0x67, 0xb3, 0xc8, 0x7f,
                                      0x04, 0xd0, 0xab, 0xe6, 0x9d, 0x49, 0x32, 0x8d, 0xf6, 0x22, 0x59, 0x14, 0x6f, 0xbb, 0xc0, 0x77, 0x0c, 0xd8, 0xa3, 0xee, 0x95, 0x41, 0x3a, 0x0b, 0x70,
                                      0xa4, 0xdf, 0x92, 0xe9, 0x3d, 0x46, 0xf1, 0x8a, 0x5e, 0x25, 0x68, 0x13, 0xc7, 0xbc, 0x61, 0x1a, 0xce, 0xb5, 0xf8, 0x83, 0x57, 0x2c, 0x9b, 0xe0, 0x34,
                                      0x4f, 0x02, 0x79, 0xad, 0xd6, 0xe7, 0x9c, 0x48, 0x33, 0x7e, 0x05, 0xd1, 0xaa, 0x1d, 0x66, 0xb2, 0xc9, 0x84, 0xff, 0x2b, 0x50, 0x62, 0x19, 0xcd, 0xb6,
                                      0xfb, 0x80, 0x54, 0x2f, 0x98, 0xe3, 0x37, 0x4c, 0x01, 0x7a, 0xae, 0xd5, 0xe4, 0x9f, 0x4b, 0x30, 0x7d, 0x06, 0xd2, 0xa9, 0x1e, 0x65, 0xb1, 0xca, 0x87,
                                      0xfc, 0x28, 0x53, 0x8e, 0xf5, 0x21, 0x5a, 0x17, 0x6c, 0xb8, 0xc3, 0x74, 0x0f, 0xdb, 0xa0, 0xed, 0x96, 0x42, 0x39, 0x08, 0x73, 0xa7, 0xdc, 0x91, 0xea,
                                      0x3e, 0x45, 0xf2, 0x89, 0x5d, 0x26, 0x6b, 0x10, 0xc4, 0xbf};

    unsigned char FromDualBasis[256] = {0x00, 0xcc, 0xac, 0x60, 0x79, 0xb5, 0xd5, 0x19, 0xf0, 0x3c, 0x5c, 0x90, 0x89, 0x45, 0x25, 0xe9, 0xfd, 0x31, 0x51, 0x9d,
                                        0x84, 0x48, 0x28, 0xe4, 0x0d, 0xc1, 0xa1, 0x6d, 0x74, 0xb8, 0xd8, 0x14, 0x2e, 0xe2, 0x82, 0x4e, 0x57, 0x9b, 0xfb, 0x37, 0xde, 0x12, 0x72, 0xbe, 0xa7,
                                        0x6b, 0x0b, 0xc7, 0xd3, 0x1f, 0x7f, 0xb3, 0xaa, 0x66, 0x06, 0xca, 0x23, 0xef, 0x8f, 0x43, 0x5a, 0x96, 0xf6, 0x3a, 0x42, 0x8e, 0xee, 0x22, 0x3b, 0xf7,
                                        0x97, 0x5b, 0xb2, 0x7e, 0x1e, 0xd2, 0xcb, 0x07, 0x67, 0xab, 0xbf, 0x73, 0x13, 0xdf, 0xc6, 0x0a, 0x6a, 0xa6, 0x4f, 0x83, 0xe3, 0x2f, 0x36, 0xfa, 0x9a,
                                        0x56, 0x6c, 0xa0, 0xc0, 0x0c, 0x15, 0xd9, 0xb9, 0x75, 0x9c, 0x50, 0x30, 0xfc, 0xe5, 0x29, 0x49, 0x85, 0x91, 0x5d, 0x3d, 0xf1, 0xe8, 0x24, 0x44, 0x88,
                                        0x61, 0xad, 0xcd, 0x01, 0x18, 0xd4, 0xb4, 0x78, 0xc5, 0x09, 0x69, 0xa5, 0xbc, 0x70, 0x10, 0xdc, 0x35, 0xf9, 0x99, 0x55, 0x4c, 0x80, 0xe0, 0x2c, 0x38,
                                        0xf4, 0x94, 0x58, 0x41, 0x8d, 0xed, 0x21, 0xc8, 0x04, 0x64, 0xa8, 0xb1, 0x7d, 0x1d, 0xd1, 0xeb, 0x27, 0x47, 0x8b, 0x92, 0x5e, 0x3e, 0xf2, 0x1b, 0xd7,
                                        0xb7, 0x7b, 0x62, 0xae, 0xce, 0x02, 0x16, 0xda, 0xba, 0x76, 0x6f, 0xa3, 0xc3, 0x0f, 0xe6, 0x2a, 0x4a, 0x86, 0x9f, 0x53, 0x33, 0xff, 0x87, 0x4b, 0x2b,
                                        0xe7, 0xfe, 0x32, 0x52, 0x9e, 0x77, 0xbb, 0xdb, 0x17, 0x0e, 0xc2, 0xa2, 0x6e, 0x7a, 0xb6, 0xd6, 0x1a, 0x03, 0xcf, 0xaf, 0x63, 0x8a, 0x46, 0x26, 0xea,
                                        0xf3, 0x3f, 0x5f, 0x93, 0xa9, 0x65, 0x05, 0xc9, 0xd0, 0x1c, 0x7c, 0xb0, 0x59, 0x95, 0xf5, 0x39, 0x20, 0xec, 0x8c, 0x40, 0x54, 0x98, 0xf8, 0x34, 0x2d,
                                        0xe1, 0x81, 0x4d, 0xa4, 0x68, 0x08, 0xc4, 0xdd, 0x11, 0x71, 0xbd};

    ReedSolomon::ReedSolomon(RS_TYPE type, int fill_bytes)
    {
        if (type == RS223)
        {
            rs = correct_reed_solomon_create(correct_rs_primitive_polynomial_ccsds, 112, 11, 32);
            d_coded_bits = 223;
            d_parity_bits = 32;
        }
        else if (type == RS239)
        {
            rs = correct_reed_solomon_create(correct_rs_primitive_polynomial_ccsds, 120, 11, 16);
            d_coded_bits = 239;
            d_parity_bits = 16;
        }

        this->fill_bytes = fill_bytes;
    }

    ReedSolomon::~ReedSolomon()
    {
        correct_reed_solomon_destroy(rs);
    }

    void ReedSolomon::decode_interlaved(uint8_t *data, bool ccsds, int i, int *errors)
    {
        for (int b = 0; b < i; b++)
        {
            deinterleave(data, buff, b, i);
            errors[b] = decode(buff, ccsds);
            interleave(buff, data, b, i);
        }
    }

    int ReedSolomon::decode(uint8_t *data, bool ccsds)
    {
        if (fill_bytes != -1) // Depuncture
        {
            memmove(&data[fill_bytes], &data[0], 255 - fill_bytes); // Move RS to the end
            memset(&data[0], 0, fill_bytes);                        // Set fill to 0
        }

        if (ccsds)
        {
            for (int i = 0; i < 255; i++)
                data[i] = FromDualBasis[data[i]];
        }

        int err = correct_reed_solomon_decode(this->rs, data, 255, odata);

        if (err == -1)
        {
            if (ccsds)
            {
                for (int i = 0; i < 255; i++)
                    data[i] = ToDualBasis[data[i]];
            }

            if (fill_bytes != -1)                                       // Repuncture
                memmove(&data[0], &data[fill_bytes], 255 - fill_bytes); // Move back parity

            return -1;
        }
        else
        {
            err = 0;

            // Calculate wrong bytes
            for (int i = 0; i < d_coded_bits; i++)
            {
                if ((data[i] ^ odata[i]) != 0)
                    err++;
            }
        }

        std::memcpy(data, odata, fill_bytes == -1 ? d_coded_bits : (d_coded_bits - fill_bytes));

        if (ccsds)
        {
            for (int i = 0; i < 255; i++)
                data[i] = ToDualBasis[data[i]];
        }

        if (fill_bytes != -1)                                       // Repuncture
            memmove(&data[0], &data[fill_bytes], 255 - fill_bytes); // Move back parity

        return err;
    }

    void ReedSolomon::encode_interlaved(uint8_t *data, bool ccsds, int i)
    {
        for (int b = 0; b < i; b++)
        {
            deinterleave(data, buff, b, i);
            encode(buff, ccsds);
            interleave(buff, data, b, i);
        }
    }

    void ReedSolomon::encode(uint8_t *data, bool ccsds)
    {
        if (ccsds)
        {
            for (int ii = 0; ii < 255; ii++)
                data[ii] = FromDualBasis[data[ii]];
        }

        correct_reed_solomon_encode(rs, data, d_coded_bits, data);

        if (ccsds)
        {
            for (int ii = 0; ii < 255; ii++)
                data[ii] = ToDualBasis[data[ii]];
        }
    }

    void ReedSolomon::deinterleave(uint8_t *data, uint8_t *output, uint8_t pos, uint8_t i)
    {
        for (int ii = 0; ii < 255 - fill_bytes; ii++)
            output[ii] = data[ii * i + pos];
    }

    void ReedSolomon::interleave(uint8_t *data, uint8_t *output, uint8_t pos, uint8_t i)
    {
        for (int ii = 0; ii < 255 - fill_bytes; ii++)
            output[ii * i + pos] = data[ii];
    }
};