// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/common/printing/print_media_l10n.h"

#include <string>
#include <string_view>

#include "base/containers/contains.h"
#include "base/containers/fixed_flat_map.h"
#include "base/i18n/number_formatting.h"
#include "base/i18n/string_compare.h"
#include "base/strings/strcat.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/utf_string_conversions.h"
#include "components/device_event_log/device_event_log.h"
#include "components/strings/grit/components_strings.h"
#include "printing/backend/print_backend_utils.h"
#include "printing/units.h"
#include "ui/base/l10n/l10n_util.h"

namespace printing {

namespace {

// Return the localized PWG name, display name, and sort group of a media name
// specified by `size` if any is found - else return an empty string in the
// named sizes group. The static map contained here is intended to reach all
// translated media names - see print_media_resources.grd.
MediaSizeInfo InfoForStandardSize(const gfx::Size& size) {
  struct RegisteredMediaInfo {
    std::string_view vendor_id;
    int l10n_id;
    MediaSizeGroup sort_group;
  };

  static constexpr auto kMediaMap = base::MakeFixedFlatMap<gfx::Size,
                                                           RegisteredMediaInfo>(
      {
          {{2600, 3700},
           {"iso_a10_26x37mm", PRINT_PREVIEW_MEDIA_ISO_A10_26X37MM,
            MediaSizeGroup::kSizeNamed}},
          {{2800, 4000},
           {"iso_c10_28x40mm", PRINT_PREVIEW_MEDIA_ISO_C10_28X40MM,
            MediaSizeGroup::kSizeNamed}},
          {{3100, 4400},
           {"iso_b10_31x44mm", PRINT_PREVIEW_MEDIA_ISO_B10_31X44MM,
            MediaSizeGroup::kSizeNamed}},
          {{3200, 4500},
           {"jis_b10_32x45mm", PRINT_PREVIEW_MEDIA_JIS_B10_32X45MM,
            MediaSizeGroup::kSizeNamed}},
          {{3700, 5200},
           {"iso_a9_37x52mm", PRINT_PREVIEW_MEDIA_ISO_A9_37X52MM,
            MediaSizeGroup::kSizeNamed}},
          {{4000, 5700},
           {"iso_c9_40x57mm", PRINT_PREVIEW_MEDIA_ISO_C9_40X57MM,
            MediaSizeGroup::kSizeNamed}},
          {{4400, 6200},
           {"iso_b9_44x62mm", PRINT_PREVIEW_MEDIA_ISO_B9_44X62MM,
            MediaSizeGroup::kSizeNamed}},
          {{4500, 6400},
           {"jis_b9_45x64mm", PRINT_PREVIEW_MEDIA_JIS_B9_45X64MM,
            MediaSizeGroup::kSizeNamed}},
          {{5080, 8890},
           {"oe_business-card_2x3.5in",
            PRINT_PREVIEW_MEDIA_OE_BUSINESS_CARD_2X3_5IN,
            MediaSizeGroup::kSizeIn}},
          {{5200, 7400},
           {"iso_a8_52x74mm", PRINT_PREVIEW_MEDIA_ISO_A8_52X74MM,
            MediaSizeGroup::kSizeNamed}},
          {{5398, 8560},
           {"iso_id-1_53.98x85.6mm", PRINT_PREVIEW_MEDIA_ISO_ID_1_53_98X85_6MM,
            MediaSizeGroup::kSizeNamed}},
          {{5400, 8600},
           {"om_card_54x86mm", PRINT_PREVIEW_MEDIA_OM_CARD_54X86MM,
            MediaSizeGroup::kSizeMm}},
          {{5500, 8500},
           {"om_business-card_55x85mm",
            PRINT_PREVIEW_MEDIA_OM_BUSINESS_CARD_55X85MM,
            MediaSizeGroup::kSizeMm}},
          {{5500, 9100},
           {"om_business-card_55x91mm",
            PRINT_PREVIEW_MEDIA_OM_BUSINESS_CARD_55X91MM,
            MediaSizeGroup::kSizeMm}},
          {{5700, 8100},
           {"iso_c8_57x81mm", PRINT_PREVIEW_MEDIA_ISO_C8_57X81MM,
            MediaSizeGroup::kSizeNamed}},
          {{6200, 8800},
           {"iso_b8_62x88mm", PRINT_PREVIEW_MEDIA_ISO_B8_62X88MM,
            MediaSizeGroup::kSizeNamed}},
          {{6400, 9100},
           {"jis_b8_64x91mm", PRINT_PREVIEW_MEDIA_JIS_B8_64X91MM,
            MediaSizeGroup::kSizeNamed}},
          {{7400, 10500},
           {"iso_a7_74x105mm", PRINT_PREVIEW_MEDIA_ISO_A7_74X105MM,
            MediaSizeGroup::kSizeNamed}},
          {{7620, 12700},
           {"na_index-3x5_3x5in", PRINT_PREVIEW_MEDIA_NA_INDEX_3X5_3X5IN,
            MediaSizeGroup::kSizeIn}},
          {{8100, 11400},
           {"iso_c7_81x114mm", PRINT_PREVIEW_MEDIA_ISO_C7_81X114MM,
            MediaSizeGroup::kSizeNamed}},
          {{8100, 16200},
           {"iso_c7c6_81x162mm", PRINT_PREVIEW_MEDIA_ISO_C7C6_81X162MM,
            MediaSizeGroup::kSizeNamed}},
          {{8800, 12500},
           {"iso_b7_88x125mm", PRINT_PREVIEW_MEDIA_ISO_B7_88X125MM,
            MediaSizeGroup::kSizeNamed}},
          {{8890, 12700},
           {"oe_photo-l_3.5x5in", PRINT_PREVIEW_MEDIA_OE_PHOTO_L_3_5X5IN,
            MediaSizeGroup::kSizeIn}},
          {{8900, 8900},
           {"om_square-photo_89x89mm",
            PRINT_PREVIEW_MEDIA_OM_SQUARE_PHOTO_89X89MM,
            MediaSizeGroup::kSizeMm}},
          {{8900, 11900},
           {"om_dsc-photo_89x119mm", PRINT_PREVIEW_MEDIA_OM_DSC_PHOTO_89X119MM,
            MediaSizeGroup::kSizeNamed}},
          {{9000, 20500},
           {"jpn_chou4_90x205mm", PRINT_PREVIEW_MEDIA_JPN_CHOU4_90X205MM,
            MediaSizeGroup::kSizeNamed}},
          {{9000, 22500},
           {"jpn_chou40_90x225mm", PRINT_PREVIEW_MEDIA_JPN_CHOU40_90X225MM,
            MediaSizeGroup::kSizeNamed}},
          {{9100, 12800},
           {"jis_b7_91x128mm", PRINT_PREVIEW_MEDIA_JIS_B7_91X128MM,
            MediaSizeGroup::kSizeNamed}},
          {{9207, 16510},
           {"na_personal_3.625x6.5in",
            PRINT_PREVIEW_MEDIA_NA_PERSONAL_3_625X6_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{9700, 15100},
           {"prc_32k_97x151mm", PRINT_PREVIEW_MEDIA_PRC_32K_97X151MM,
            MediaSizeGroup::kSizeNamed}},
          {{9800, 19000},
           {"jpn_you6_98x190mm", PRINT_PREVIEW_MEDIA_JPN_YOU6_98X190MM,
            MediaSizeGroup::kSizeNamed}},
          {{9842, 19050},
           {"na_monarch_3.875x7.5in",
            PRINT_PREVIEW_MEDIA_NA_MONARCH_3_875X7_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{9842, 22542},
           {"na_number-9_3.875x8.875in",
            PRINT_PREVIEW_MEDIA_NA_NUMBER_9_3_875X8_875IN,
            MediaSizeGroup::kSizeNamed}},
          {{10000, 14800},
           {"jpn_hagaki_100x148mm", PRINT_PREVIEW_MEDIA_JPN_HAGAKI_100X148MM,
            MediaSizeGroup::kSizeNamed}},
          {{10000, 15000},
           {"om_small-photo_100x150mm",
            PRINT_PREVIEW_MEDIA_OM_SMALL_PHOTO_100X150MM,
            MediaSizeGroup::kSizeMm}},
          {{10000, 20000},
           {"om_wide-photo_100x200mm",
            PRINT_PREVIEW_MEDIA_OM_WIDE_PHOTO_100X200MM,
            MediaSizeGroup::kSizeMm}},
          {{10160, 10160},
           {"oe_square-photo_4x4in", PRINT_PREVIEW_MEDIA_OE_SQUARE_PHOTO_4X4IN,
            MediaSizeGroup::kSizeIn}},
          {{10160, 15240},
           {"na_index-4x6_4x6in", PRINT_PREVIEW_MEDIA_NA_INDEX_4X6_4X6IN,
            MediaSizeGroup::kSizeIn}},
          {{10200, 16500},
           {"prc_1_102x165mm", PRINT_PREVIEW_MEDIA_PRC_1_102X165MM,
            MediaSizeGroup::kSizeNamed}},
          {{10200, 17600},
           {"prc_2_102x176mm", PRINT_PREVIEW_MEDIA_PRC_2_102X176MM,
            MediaSizeGroup::kSizeNamed}},
          {{10477, 24130},
           {"na_number-10_4.125x9.5in",
            PRINT_PREVIEW_MEDIA_NA_NUMBER_10_4_125X9_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{10500, 14800},
           {"iso_a6_105x148mm", PRINT_PREVIEW_MEDIA_ISO_A6_105X148MM,
            MediaSizeGroup::kSizeNamed}},
          {{10500, 23500},
           {"jpn_you4_105x235mm", PRINT_PREVIEW_MEDIA_JPN_YOU4_105X235MM,
            MediaSizeGroup::kSizeNamed}},
          {{11000, 20800},
           {"prc_4_110x208mm", PRINT_PREVIEW_MEDIA_PRC_4_110X208MM,
            MediaSizeGroup::kSizeNamed}},
          {{11000, 22000},
           {"iso_dl_110x220mm", PRINT_PREVIEW_MEDIA_ISO_DL_110X220MM,
            MediaSizeGroup::kSizeNamed}},
          {{11000, 23000},
           {"om_italian_110x230mm", PRINT_PREVIEW_MEDIA_OM_ITALIAN_110X230MM,
            MediaSizeGroup::kSizeNamed}},
          {{11110, 14600},
           {"jpn_chou2_111.1x146mm", PRINT_PREVIEW_MEDIA_JPN_CHOU2_111_1X146MM,
            MediaSizeGroup::kSizeNamed}},
          {{11112, 14605},
           {"na_a2_4.375x5.75in", PRINT_PREVIEW_MEDIA_NA_A2_4_375X5_75IN,
            MediaSizeGroup::kSizeNamed}},
          {{11400, 16200},
           {"iso_c6_114x162mm", PRINT_PREVIEW_MEDIA_ISO_C6_114X162MM,
            MediaSizeGroup::kSizeNamed}},
          {{11400, 22900},
           {"iso_c6c5_114x229mm", PRINT_PREVIEW_MEDIA_ISO_C6C5_114X229MM,
            MediaSizeGroup::kSizeNamed}},
          {{11430, 26352},
           {"na_number-11_4.5x10.375in",
            PRINT_PREVIEW_MEDIA_NA_NUMBER_11_4_5X10_375IN,
            MediaSizeGroup::kSizeNamed}},
          {{11900, 19700},
           {"jpn_kaku8_119x197mm", PRINT_PREVIEW_MEDIA_JPN_KAKU8_119X197MM,
            MediaSizeGroup::kSizeNamed}},
          {{12000, 23500},
           {"jpn_chou3_120x235mm", PRINT_PREVIEW_MEDIA_JPN_CHOU3_120X235MM,
            MediaSizeGroup::kSizeNamed}},
          {{12000, 30900},
           {"prc_8_120x309mm", PRINT_PREVIEW_MEDIA_PRC_8_120X309MM,
            MediaSizeGroup::kSizeNamed}},
          {{12000, 32000},
           {"prc_6_120x320mm", PRINT_PREVIEW_MEDIA_PRC_6_120X320MM,
            MediaSizeGroup::kSizeNamed}},
          {{12065, 27940},
           {"na_number-12_4.75x11in",
            PRINT_PREVIEW_MEDIA_NA_NUMBER_12_4_75X11IN,
            MediaSizeGroup::kSizeNamed}},
          {{12500, 17600},
           {"iso_b6_125x176mm", PRINT_PREVIEW_MEDIA_ISO_B6_125X176MM,
            MediaSizeGroup::kSizeNamed}},
          {{12500, 32400},
           {"iso_b6c4_125x324mm", PRINT_PREVIEW_MEDIA_ISO_B6C4_125X324MM,
            MediaSizeGroup::kSizeNamed}},
          {{12700, 12700},
           {"oe_square-photo_5x5in", PRINT_PREVIEW_MEDIA_OE_SQUARE_PHOTO_5X5IN,
            MediaSizeGroup::kSizeIn}},
          {{12700, 17780},
           {"na_5x7_5x7in", PRINT_PREVIEW_MEDIA_NA_5X7_5X7IN,
            MediaSizeGroup::kSizeIn}},
          {{12700, 20320},
           {"na_index-5x8_5x8in", PRINT_PREVIEW_MEDIA_NA_INDEX_5X8_5X8IN,
            MediaSizeGroup::kSizeIn}},
          {{12700, 29210},
           {"na_number-14_5x11.5in", PRINT_PREVIEW_MEDIA_NA_NUMBER_14_5X11_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{12800, 18200},
           {"jis_b6_128x182mm", PRINT_PREVIEW_MEDIA_JIS_B6_128X182MM,
            MediaSizeGroup::kSizeNamed}},
          {{13000, 18000},
           {"om_medium-photo_130x180mm",
            PRINT_PREVIEW_MEDIA_OM_MEDIUM_PHOTO_130X180MM,
            MediaSizeGroup::kSizeMm}},
          {{13970, 21590},
           {"na_invoice_5.5x8.5in", PRINT_PREVIEW_MEDIA_NA_INVOICE_5_5X8_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{14200, 20500},
           {"jpn_kaku7_142x205mm", PRINT_PREVIEW_MEDIA_JPN_KAKU7_142X205MM,
            MediaSizeGroup::kSizeNamed}},
          {{14600, 21500},
           {"prc_16k_146x215mm", PRINT_PREVIEW_MEDIA_PRC_16K_146X215MM,
            MediaSizeGroup::kSizeNamed}},
          {{14800, 20000},
           {"jpn_oufuku_148x200mm", PRINT_PREVIEW_MEDIA_JPN_OUFUKU_148X200MM,
            MediaSizeGroup::kSizeNamed}},
          {{14800, 21000},
           {"iso_a5_148x210mm", PRINT_PREVIEW_MEDIA_ISO_A5_148X210MM,
            MediaSizeGroup::kSizeNamed}},
          {{15240, 20320},
           {"na_index-4x6-ext_6x8in",
            PRINT_PREVIEW_MEDIA_NA_INDEX_4X6_EXT_6X8IN,
            MediaSizeGroup::kSizeIn}},
          {{15240, 22860},
           {"na_6x9_6x9in", PRINT_PREVIEW_MEDIA_NA_6X9_6X9IN,
            MediaSizeGroup::kSizeNamed}},
          {{16000, 23000},
           {"prc_7_160x230mm", PRINT_PREVIEW_MEDIA_PRC_7_160X230MM,
            MediaSizeGroup::kSizeNamed}},
          {{16200, 22900},
           {"iso_c5_162x229mm", PRINT_PREVIEW_MEDIA_ISO_C5_162X229MM,
            MediaSizeGroup::kSizeNamed}},
          {{16510, 24130},
           {"na_c5_6.5x9.5in", PRINT_PREVIEW_MEDIA_NA_C5_6_5X9_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{17400, 23500},
           {"iso_a5-extra_174x235mm",
            PRINT_PREVIEW_MEDIA_ISO_A5_EXTRA_174X235MM,
            MediaSizeGroup::kSizeNamed}},
          {{17600, 25000},
           {"iso_b5_176x250mm", PRINT_PREVIEW_MEDIA_ISO_B5_176X250MM,
            MediaSizeGroup::kSizeNamed}},
          {{17780, 22860},
           {"na_7x9_7x9in", PRINT_PREVIEW_MEDIA_NA_7X9_7X9IN,
            MediaSizeGroup::kSizeNamed}},
          {{18200, 25700},
           {"jis_b5_182x257mm", PRINT_PREVIEW_MEDIA_JIS_B5_182X257MM,
            MediaSizeGroup::kSizeNamed}},
          {{18400, 26000},
           {"om_16k_184x260mm", PRINT_PREVIEW_MEDIA_OM_16K_184X260MM,
            MediaSizeGroup::kSizeMm}},
          {{18415, 26670},
           {"na_executive_7.25x10.5in",
            PRINT_PREVIEW_MEDIA_NA_EXECUTIVE_7_25X10_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{19000, 24000},
           {"jpn_kaku5_190x240mm", PRINT_PREVIEW_MEDIA_JPN_KAKU5_190X240MM,
            MediaSizeGroup::kSizeNamed}},
          {{19500, 27000},
           {"om_16k_195x270mm", PRINT_PREVIEW_MEDIA_OM_16K_195X270MM,
            MediaSizeGroup::kSizeMm}},
          {{19685, 27305},
           {"roc_16k_7.75x10.75in", PRINT_PREVIEW_MEDIA_ROC_16K_7_75X10_75IN,
            MediaSizeGroup::kSizeNamed}},
          {{19700, 26700},
           {"jpn_kaku4_197x267mm", PRINT_PREVIEW_MEDIA_JPN_KAKU4_197X267MM,
            MediaSizeGroup::kSizeNamed}},
          {{19800, 27500},
           {"om_juuro-ku-kai_198x275mm",
            PRINT_PREVIEW_MEDIA_OM_JUURO_KU_KAI_198X275MM,
            MediaSizeGroup::kSizeMm}},
          {{20000, 30000},
           {"om_large-photo_200x300mm",
            PRINT_PREVIEW_MEDIA_OM_LARGE_PHOTO_200X300,
            MediaSizeGroup::kSizeMm}},
          {{20100, 27600},
           {"iso_b5-extra_201x276mm",
            PRINT_PREVIEW_MEDIA_ISO_B5_EXTRA_201X276MM,
            MediaSizeGroup::kSizeNamed}},
          {{20320, 25400},
           {"na_govt-letter_8x10in", PRINT_PREVIEW_MEDIA_NA_GOVT_LETTER_8X10IN,
            MediaSizeGroup::kSizeIn}},
          {{20320, 30480},
           {"oe_photo-s8r_8x12in", PRINT_PREVIEW_MEDIA_OE_PHOTO_S8R_8X12IN,
            MediaSizeGroup::kSizeIn}},
          {{20320, 33020},
           {"na_govt-legal_8x13in", PRINT_PREVIEW_MEDIA_NA_GOVT_LEGAL_8X13IN,
            MediaSizeGroup::kSizeIn}},
          {{21000, 29700},
           {"iso_a4_210x297mm", PRINT_PREVIEW_MEDIA_ISO_A4_210X297MM,
            MediaSizeGroup::kSizeNamed}},
          {{21000, 33000},
           {"om_folio_210x330mm", PRINT_PREVIEW_MEDIA_OM_FOLIO_210X330MM,
            MediaSizeGroup::kSizeMm}},
          {{21500, 30500},
           {"iso_ra4_215x305mm", PRINT_PREVIEW_MEDIA_ISO_RA4_215X305MM,
            MediaSizeGroup::kSizeNamed}},
          {{21500, 31500},
           {"om_folio-sp_215x315mm", PRINT_PREVIEW_MEDIA_OM_FOLIO_SP_215X315MM,
            MediaSizeGroup::kSizeMm}},
          {{21590, 27508},
           {"na_quarto_8.5x10.83in", PRINT_PREVIEW_MEDIA_NA_QUARTO_8_5X10_83IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 27940},
           {"na_letter_8.5x11in", PRINT_PREVIEW_MEDIA_NA_LETTER_8_5X11IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 30480},
           {"na_fanfold-eur_8.5x12in",
            PRINT_PREVIEW_MEDIA_NA_FANFOLD_EUR_8_5X12IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 32232},
           {"na_letter-plus_8.5x12.69in",
            PRINT_PREVIEW_MEDIA_NA_LETTER_PLUS_8_5X12_69IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 33020},
           {"na_foolscap_8.5x13in", PRINT_PREVIEW_MEDIA_NA_FOOLSCAP_8_5X13IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 34036},
           {"na_oficio_8.5x13.4in", PRINT_PREVIEW_MEDIA_NA_OFICIO_8_5X13_4IN,
            MediaSizeGroup::kSizeNamed}},
          {{21590, 35560},
           {"na_legal_8.5x14in", PRINT_PREVIEW_MEDIA_NA_LEGAL_8_5X14IN,
            MediaSizeGroup::kSizeNamed}},
          {{21600, 27700},
           {"jpn_kaku3_216x277mm", PRINT_PREVIEW_MEDIA_JPN_KAKU3_216X277MM,
            MediaSizeGroup::kSizeNamed}},
          {{21600, 33000},
           {"jis_exec_216x330mm", PRINT_PREVIEW_MEDIA_JIS_EXEC_216X330MM,
            MediaSizeGroup::kSizeNamed}},
          {{22000, 22000},
           {"om_invite_220x220mm", PRINT_PREVIEW_MEDIA_OM_INVITE_220X220MM,
            MediaSizeGroup::kSizeNamed}},
          {{22500, 29700},
           {"iso_a4-tab_225x297mm", PRINT_PREVIEW_MEDIA_ISO_A4_TAB_225X297MM,
            MediaSizeGroup::kSizeNamed}},
          {{22500, 32000},
           {"iso_sra4_225x320mm", PRINT_PREVIEW_MEDIA_ISO_SRA4_225X320MM,
            MediaSizeGroup::kSizeNamed}},
          {{22707, 35560},
           {"na_super-a_8.94x14in", PRINT_PREVIEW_MEDIA_NA_SUPER_A_8_94X14IN,
            MediaSizeGroup::kSizeNamed}},
          {{22860, 27940},
           {"na_9x11_9x11in", PRINT_PREVIEW_MEDIA_NA_9X11_9X11IN,
            MediaSizeGroup::kSizeNamed}},
          {{22860, 30480},
           {"na_arch-a_9x12in", PRINT_PREVIEW_MEDIA_NA_ARCH_A_9X12IN,
            MediaSizeGroup::kSizeNamed}},
          {{22900, 32400},
           {"iso_c4_229x324mm", PRINT_PREVIEW_MEDIA_ISO_C4_229X324MM,
            MediaSizeGroup::kSizeNamed}},
          {{23550, 32230},
           {"iso_a4-extra_235.5x322.3mm",
            PRINT_PREVIEW_MEDIA_ISO_A4_EXTRA_235_5X322_3MM,
            MediaSizeGroup::kSizeNamed}},
          {{24000, 32210},
           {"jpn_kahu_240x322.1mm", PRINT_PREVIEW_MEDIA_JPN_KAHU_240X322_1MM,
            MediaSizeGroup::kSizeNamed}},
          {{24000, 33200},
           {"jpn_kaku2_240x332mm", PRINT_PREVIEW_MEDIA_JPN_KAKU2_240X332MM,
            MediaSizeGroup::kSizeNamed}},
          {{24130, 30480},
           {"na_letter-extra_9.5x12in",
            PRINT_PREVIEW_MEDIA_NA_LETTER_EXTRA_9_5X12IN,
            MediaSizeGroup::kSizeNamed}},
          {{24130, 38100},
           {"na_legal-extra_9.5x15in",
            PRINT_PREVIEW_MEDIA_NA_LEGAL_EXTRA_9_5X15IN,
            MediaSizeGroup::kSizeNamed}},
          {{25000, 35300},
           {"iso_b4_250x353mm", PRINT_PREVIEW_MEDIA_ISO_B4_250X353MM,
            MediaSizeGroup::kSizeNamed}},
          {{25400, 27940},
           {"na_10x11_10x11in", PRINT_PREVIEW_MEDIA_NA_10X11_10X11IN,
            MediaSizeGroup::kSizeIn}},
          {{25400, 30480},
           {"oe_photo-10r_10x12in", PRINT_PREVIEW_MEDIA_OE_PHOTO_10R_10X12IN,
            MediaSizeGroup::kSizeIn}},
          {{25400, 33020},
           {"na_10x13_10x13in", PRINT_PREVIEW_MEDIA_NA_10X13_10X13IN,
            MediaSizeGroup::kSizeIn}},
          {{25400, 35560},
           {"na_10x14_10x14in", PRINT_PREVIEW_MEDIA_NA_10X14_10X14IN,
            MediaSizeGroup::kSizeIn}},
          {{25400, 38100},
           {"na_10x15_10x15in", PRINT_PREVIEW_MEDIA_NA_10X15_10X15IN,
            MediaSizeGroup::kSizeIn}},
          {{25700, 36400},
           {"jis_b4_257x364mm", PRINT_PREVIEW_MEDIA_JIS_B4_257X364MM,
            MediaSizeGroup::kSizeNamed}},
          {{26700, 38900},
           {"om_pa-kai_267x389mm", PRINT_PREVIEW_MEDIA_OM_PA_KAI_267X389MM,
            MediaSizeGroup::kSizeMm}},
          {{27000, 38200},
           {"jpn_kaku1_270x382mm", PRINT_PREVIEW_MEDIA_JPN_KAKU1_270X382MM,
            MediaSizeGroup::kSizeNamed}},
          {{27305, 39370},
           {"roc_8k_10.75x15.5in", PRINT_PREVIEW_MEDIA_ROC_8K_10_75X15_5IN,
            MediaSizeGroup::kSizeNamed}},
          {{27500, 39500},
           {"om_dai-pa-kai_275x395mm",
            PRINT_PREVIEW_MEDIA_OM_DAI_PA_KAI_275X395MM,
            MediaSizeGroup::kSizeMm}},
          {{27940, 30480},
           {"na_11x12_11x12in", PRINT_PREVIEW_MEDIA_NA_11X12_11X12IN,
            MediaSizeGroup::kSizeIn}},
          {{27940, 35560},
           {"na_edp_11x14in", PRINT_PREVIEW_MEDIA_NA_EDP_11X14IN,
            MediaSizeGroup::kSizeNamed}},
          {{27940, 37782},
           {"na_fanfold-us_11x14.875in",
            PRINT_PREVIEW_MEDIA_NA_FANFOLD_US_11X14_875IN,
            MediaSizeGroup::kSizeNamed}},
          {{27940, 38100},
           {"na_11x15_11x15in", PRINT_PREVIEW_MEDIA_NA_11X15_11X15IN,
            MediaSizeGroup::kSizeIn}},
          {{27940, 43180},
           {"na_ledger_11x17in", PRINT_PREVIEW_MEDIA_NA_LEDGER_11X17IN,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 42000},
           {"iso_a3_297x420mm", PRINT_PREVIEW_MEDIA_ISO_A3_297X420MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 63000},
           {"iso_a4x3_297x630mm", PRINT_PREVIEW_MEDIA_ISO_A4X3_297X630MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 84100},
           {"iso_a4x4_297x841mm", PRINT_PREVIEW_MEDIA_ISO_A4X4_297X841MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 105100},
           {"iso_a4x5_297x1051mm", PRINT_PREVIEW_MEDIA_ISO_A4X5_297X1051MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 126100},
           {"iso_a4x6_297x1261mm", PRINT_PREVIEW_MEDIA_ISO_A4X6_297X1261MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 147100},
           {"iso_a4x7_297x1471mm", PRINT_PREVIEW_MEDIA_ISO_A4X7_297X1471MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 168200},
           {"iso_a4x8_297x1682mm", PRINT_PREVIEW_MEDIA_ISO_A4X8_297X1682MM,
            MediaSizeGroup::kSizeNamed}},
          {{29700, 189200},
           {"iso_a4x9_297x1892mm", PRINT_PREVIEW_MEDIA_ISO_A4X9_297X1892MM,
            MediaSizeGroup::kSizeNamed}},
          {{30000, 40000},
           {"om_photo-30x40_300x400mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_30X40_300X400MM,
            MediaSizeGroup::kSizeMm}},
          {{30000, 45000},
           {"om_photo-30x45_300x450mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_30X45_300X450MM,
            MediaSizeGroup::kSizeMm}},
          {{30480, 35560},
           {"na_eur-edp_12x14in", PRINT_PREVIEW_MEDIA_NA_EUR_EDP_12X14IN,
            MediaSizeGroup::kSizeNamed}},
          {{30480, 38100},
           {"oe_photo-12r_12x15in", PRINT_PREVIEW_MEDIA_OE_PHOTO_12R_12X15IN,
            MediaSizeGroup::kSizeIn}},
          {{30480, 40640},
           {"oe_12x16_12x16in", PRINT_PREVIEW_MEDIA_OE_12X16_12X16IN,
            MediaSizeGroup::kSizeIn}},
          {{30480, 45720},
           {"na_arch-b_12x18in", PRINT_PREVIEW_MEDIA_NA_ARCH_B_12X18IN,
            MediaSizeGroup::kSizeIn}},
          {{30480, 48260},
           {"na_12x19_12x19in", PRINT_PREVIEW_MEDIA_NA_12X19_12X19IN,
            MediaSizeGroup::kSizeIn}},
          {{30480, 48691},
           {"na_b-plus_12x19.17in", PRINT_PREVIEW_MEDIA_NA_B_PLUS_12X19_17IN,
            MediaSizeGroup::kSizeNamed}},
          {{30500, 43000},
           {"iso_ra3_305x430mm", PRINT_PREVIEW_MEDIA_ISO_RA3_305X430MM,
            MediaSizeGroup::kSizeNamed}},
          {{32000, 45000},
           {"iso_sra3_320x450mm", PRINT_PREVIEW_MEDIA_ISO_SRA3_320X450MM,
            MediaSizeGroup::kSizeNamed}},
          {{32200, 44500},
           {"iso_a3-extra_322x445mm",
            PRINT_PREVIEW_MEDIA_ISO_A3_EXTRA_322X445MM,
            MediaSizeGroup::kSizeNamed}},
          {{32400, 45800},
           {"iso_c3_324x458mm", PRINT_PREVIEW_MEDIA_ISO_C3_324X458MM,
            MediaSizeGroup::kSizeNamed}},
          {{33020, 48260},
           {"na_super-b_13x19in", PRINT_PREVIEW_MEDIA_NA_SUPER_B_13X19IN,
            MediaSizeGroup::kSizeNamed}},
          {{35000, 46000},
           {"om_photo-35x46_350x460mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_35X46_350X460MM,
            MediaSizeGroup::kSizeMm}},
          {{35300, 50000},
           {"iso_b3_353x500mm", PRINT_PREVIEW_MEDIA_ISO_B3_353X500MM,
            MediaSizeGroup::kSizeNamed}},
          {{35560, 43180},
           {"oe_14x17_14x17in", PRINT_PREVIEW_MEDIA_OE_14X17_14X17IN,
            MediaSizeGroup::kSizeIn}},
          {{35560, 45720},
           {"oe_photo-14x18_14x18in",
            PRINT_PREVIEW_MEDIA_OE_PHOTO_14X18_14X18IN,
            MediaSizeGroup::kSizeIn}},
          {{36400, 51500},
           {"jis_b3_364x515mm", PRINT_PREVIEW_MEDIA_JIS_B3_364X515MM,
            MediaSizeGroup::kSizeNamed}},
          {{40000, 60000},
           {"om_photo-40x60_400x600mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_40X60_400X600MM,
            MediaSizeGroup::kSizeMm}},
          {{40640, 50800},
           {"oe_photo-16r_16x20in", PRINT_PREVIEW_MEDIA_OE_PHOTO_16R_16X20IN,
            MediaSizeGroup::kSizeIn}},
          {{42000, 59400},
           {"iso_a2_420x594mm", PRINT_PREVIEW_MEDIA_ISO_A2_420X594MM,
            MediaSizeGroup::kSizeNamed}},
          {{42000, 89100},
           {"iso_a3x3_420x891mm", PRINT_PREVIEW_MEDIA_ISO_A3X3_420X891MM,
            MediaSizeGroup::kSizeNamed}},
          {{42000, 118900},
           {"iso_a3x4_420x1189mm", PRINT_PREVIEW_MEDIA_ISO_A3X4_420X1189MM,
            MediaSizeGroup::kSizeNamed}},
          {{42000, 148600},
           {"iso_a3x5_420x1486mm", PRINT_PREVIEW_MEDIA_ISO_A3X5_420X1486MM,
            MediaSizeGroup::kSizeNamed}},
          {{42000, 178300},
           {"iso_a3x6_420x1783mm", PRINT_PREVIEW_MEDIA_ISO_A3X6_420X1783MM,
            MediaSizeGroup::kSizeNamed}},
          {{42000, 208000},
           {"iso_a3x7_420x2080mm", PRINT_PREVIEW_MEDIA_ISO_A3X7_420X2080MM,
            MediaSizeGroup::kSizeNamed}},
          {{43000, 61000},
           {"iso_ra2_430x610mm", PRINT_PREVIEW_MEDIA_ISO_RA2_430X610MM,
            MediaSizeGroup::kSizeNamed}},
          {{43180, 55880},
           {"na_c_17x22in", PRINT_PREVIEW_MEDIA_NA_C_17X22IN,
            MediaSizeGroup::kSizeIn}},
          {{43180, 60960},
           {"oe_a2plus_17x24in", PRINT_PREVIEW_MEDIA_OE_A2PLUS_17X24IN,
            MediaSizeGroup::kSizeIn}},
          {{45000, 64000},
           {"iso_sra2_450x640mm", PRINT_PREVIEW_MEDIA_ISO_SRA2_450X640MM,
            MediaSizeGroup::kSizeNamed}},
          {{45720, 55880},
           {"oe_18x22_18x22in", PRINT_PREVIEW_MEDIA_OE_18X22_18X22IN,
            MediaSizeGroup::kSizeIn}},
          {{45720, 60960},
           {"na_arch-c_18x24in", PRINT_PREVIEW_MEDIA_NA_ARCH_C_18X24IN,
            MediaSizeGroup::kSizeIn}},
          {{45800, 64800},
           {"iso_c2_458x648mm", PRINT_PREVIEW_MEDIA_ISO_C2_458X648MM,
            MediaSizeGroup::kSizeNamed}},
          {{50000, 70700},
           {"iso_b2_500x707mm", PRINT_PREVIEW_MEDIA_ISO_B2_500X707MM,
            MediaSizeGroup::kSizeNamed}},
          {{50000, 75000},
           {"om_photo-50x75_500x750mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_50X75_500X750MM,
            MediaSizeGroup::kSizeMm}},
          {{50000, 76000},
           {"om_photo-50x76_500x760mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_50X76_500X760MM,
            MediaSizeGroup::kSizeMm}},
          {{50800, 60960},
           {"oe_photo-20r_20x24in", PRINT_PREVIEW_MEDIA_OE_PHOTO_20R_20X24IN,
            MediaSizeGroup::kSizeIn}},
          {{51500, 72800},
           {"jis_b2_515x728mm", PRINT_PREVIEW_MEDIA_JIS_B2_515X728MM,
            MediaSizeGroup::kSizeNamed}},
          {{55880, 71120},
           {"oe_photo-22x28_22x28in",
            PRINT_PREVIEW_MEDIA_OE_PHOTO_22X28_22X28IN,
            MediaSizeGroup::kSizeIn}},
          {{55880, 74930},
           {"oe_photo-22r_22x29.5in",
            PRINT_PREVIEW_MEDIA_OE_PHOTO_22R_22X29_5IN,
            MediaSizeGroup::kSizeIn}},
          {{55880, 86360},
           {"na_d_22x34in", PRINT_PREVIEW_MEDIA_NA_D_22X34IN,
            MediaSizeGroup::kSizeIn}},
          {{59400, 84100},
           {"iso_a1_594x841mm", PRINT_PREVIEW_MEDIA_ISO_A1_594X841MM,
            MediaSizeGroup::kSizeNamed}},
          {{59400, 126100},
           {"iso_a2x3_594x1261mm", PRINT_PREVIEW_MEDIA_ISO_A2X3_594X1261MM,
            MediaSizeGroup::kSizeNamed}},
          {{59400, 168200},
           {"iso_a2x4_594x1682mm", PRINT_PREVIEW_MEDIA_ISO_A2X4_594X1682MM,
            MediaSizeGroup::kSizeNamed}},
          {{59400, 210200},
           {"iso_a2x5_594x2102mm", PRINT_PREVIEW_MEDIA_ISO_A2X5_594X2102MM,
            MediaSizeGroup::kSizeNamed}},
          {{60000, 90000},
           {"om_photo-60x90_600x900mm",
            PRINT_PREVIEW_MEDIA_OM_PHOTO_60X90_600X900MM,
            MediaSizeGroup::kSizeMm}},
          {{60960, 76200},
           {"oe_photo-24x30_24x30in",
            PRINT_PREVIEW_MEDIA_OE_PHOTO_24X30_24X30IN,
            MediaSizeGroup::kSizeIn}},
          {{60960, 80010},
           {"oe_photo-24r_24x31.5in",
            PRINT_PREVIEW_MEDIA_OE_PHOTO_24R_24X31_5IN,
            MediaSizeGroup::kSizeIn}},
          {{60960, 91440},
           {"na_arch-d_24x36in", PRINT_PREVIEW_MEDIA_NA_ARCH_D_24X36IN,
            MediaSizeGroup::kSizeIn}},
          {{61000, 86000},
           {"iso_ra1_610x860mm", PRINT_PREVIEW_MEDIA_ISO_RA1_610X860MM,
            MediaSizeGroup::kSizeNamed}},
          {{64000, 90000},
           {"iso_sra1_640x900mm", PRINT_PREVIEW_MEDIA_ISO_SRA1_640X900MM,
            MediaSizeGroup::kSizeNamed}},
          {{64800, 91700},
           {"iso_c1_648x917mm", PRINT_PREVIEW_MEDIA_ISO_C1_648X917MM,
            MediaSizeGroup::kSizeNamed}},
          {{66040, 96520},
           {"na_arch-e2_26x38in", PRINT_PREVIEW_MEDIA_NA_ARCH_E2_26X38IN,
            MediaSizeGroup::kSizeIn}},
          {{68580, 99060},
           {"na_arch-e3_27x39in", PRINT_PREVIEW_MEDIA_NA_ARCH_E3_27X39IN,
            MediaSizeGroup::kSizeIn}},
          {{70700, 100000},
           {"iso_b1_707x1000mm", PRINT_PREVIEW_MEDIA_ISO_B1_707X1000MM,
            MediaSizeGroup::kSizeNamed}},
          {{71120, 101600},
           {"asme_f_28x40in", PRINT_PREVIEW_MEDIA_ASME_F_28X40IN,
            MediaSizeGroup::kSizeIn}},
          {{72800, 103000},
           {"jis_b1_728x1030mm", PRINT_PREVIEW_MEDIA_JIS_B1_728X1030MM,
            MediaSizeGroup::kSizeNamed}},
          {{76200, 101600},
           {"oe_photo-30r_30x40in", PRINT_PREVIEW_MEDIA_OE_PHOTO_30R_30X40IN,
            MediaSizeGroup::kSizeIn}},
          {{76200, 106680},
           {"na_wide-format_30x42in",
            PRINT_PREVIEW_MEDIA_NA_WIDE_FORMAT_30X42IN,
            MediaSizeGroup::kSizeIn}},
          {{84100, 118900},
           {"iso_a0_841x1189mm", PRINT_PREVIEW_MEDIA_ISO_A0_841X1189MM,
            MediaSizeGroup::kSizeNamed}},
          {{84100, 178300},
           {"iso_a1x3_841x1783mm", PRINT_PREVIEW_MEDIA_ISO_A1X3_841X1783MM,
            MediaSizeGroup::kSizeNamed}},
          {{84100, 237800},
           {"iso_a1x4_841x2378mm", PRINT_PREVIEW_MEDIA_ISO_A1X4_841X2378MM,
            MediaSizeGroup::kSizeNamed}},
          {{86000, 122000},
           {"iso_ra0_860x1220mm", PRINT_PREVIEW_MEDIA_ISO_RA0_860X1220MM,
            MediaSizeGroup::kSizeNamed}},
          {{86360, 111760},
           {"na_e_34x44in", PRINT_PREVIEW_MEDIA_NA_E_34X44IN,
            MediaSizeGroup::kSizeIn}},
          {{90000, 128000},
           {"iso_sra0_900x1280mm", PRINT_PREVIEW_MEDIA_ISO_SRA0_900X1280MM,
            MediaSizeGroup::kSizeNamed}},
          {{91440, 121920},
           {"na_arch-e_36x48in", PRINT_PREVIEW_MEDIA_NA_ARCH_E_36X48IN,
            MediaSizeGroup::kSizeIn}},
          {{91700, 129700},
           {"iso_c0_917x1297mm", PRINT_PREVIEW_MEDIA_ISO_C0_917X1297MM,
            MediaSizeGroup::kSizeNamed}},
          {{100000, 141400},
           {"iso_b0_1000x1414mm", PRINT_PREVIEW_MEDIA_ISO_B0_1000X1414MM,
            MediaSizeGroup::kSizeNamed}},
          {{103000, 145600},
           {"jis_b0_1030x1456mm", PRINT_PREVIEW_MEDIA_JIS_B0_1030X1456MM,
            MediaSizeGroup::kSizeNamed}},
          {{111760, 172720},
           {"na_f_44x68in", PRINT_PREVIEW_MEDIA_NA_F_44X68IN,
            MediaSizeGroup::kSizeIn}},
          {{118900, 168200},
           {"iso_2a0_1189x1682mm", PRINT_PREVIEW_MEDIA_ISO_2A0_1189X1682MM,
            MediaSizeGroup::kSizeNamed}},
          {{118900, 252300},
           {"iso_a0x3_1189x2523mm", PRINT_PREVIEW_MEDIA_ISO_A0X3_1189X2523MM,
            MediaSizeGroup::kSizeNamed}},
      },
      [](const gfx::Size& a, const gfx::Size& b) {
        auto result = a.width() - b.width();
        if (result == 0) {
          result = a.height() - b.height();
        }
        return result < 0;
      });

  auto it = kMediaMap.find(
      {size.width() / kMicronsPerPwgUnit, size.height() / kMicronsPerPwgUnit});
  return it != kMediaMap.end()
             ? MediaSizeInfo{std::string(it->second.vendor_id),
                             l10n_util::GetStringUTF16(it->second.l10n_id),
                             it->second.sort_group}
             : MediaSizeInfo{"", u"", MediaSizeGroup::kSizeNamed};
}

// Generate a vendor ID, human-readable name, and sort group from size
// information.
MediaSizeInfo InfoForUnregisteredSize(const gfx::Size& size_um) {
  int width_um = size_um.width();
  int height_um = size_um.height();

  // Generate a vendor ID so we have something to populate the field with.
  std::string vendor_id =
      base::StrCat({"om_", base::NumberToString(width_um), "x",
                    base::NumberToString(height_um), "um_",
                    base::NumberToString(width_um / kMicronsPerMm), "x",
                    base::NumberToString(height_um / kMicronsPerMm), "mm"});

  MediaSizeGroup group = MediaSizeGroup::kSizeMm;
  int message_id = PRINT_PREVIEW_MEDIA_DIMENSIONS_MM;
  int conversion_factor = kMicronsPerMm;
  int max_fractional_digits = 0;

  // Try converting to inches. If either width or height is a multiple of
  // 1/4 inch, display the size as inches. Otherwise, display the size as
  // millimeters.
  if (width_um % (kMicronsPerInch / 4) == 0 ||
      height_um % (kMicronsPerInch / 4) == 0) {
    group = MediaSizeGroup::kSizeIn;
    message_id = PRINT_PREVIEW_MEDIA_DIMENSIONS_INCHES;
    conversion_factor = kMicronsPerInch;
    max_fractional_digits = 3;
  }

  // If the width and height are in inches, display them with up to 3 digits
  // after the decimal point. 3 digits is the sweet spot where even unusual
  // fractions like sixths, eighths, and ninths are legible without showing the
  // rounding errors from their conversion to PWG units. The "up to" is
  // important since even many unregistered sizes include integer dimensions,
  // and "5 x 6.5 in" is more legible than "5.000 x 6.500 in". Even more
  // importantly, it matches how registered but unnamed sizes like "3 x 5 in"
  // are written. For millimeter sizes, round to the nearest integer, since any
  // fractional part is probably a rounding error from the mm->pt->mm conversion
  // imposed by the PPD format.
  return {vendor_id,
          l10n_util::GetStringFUTF16(
              message_id,
              base::FormatDouble(static_cast<double>(width_um) /
                                     static_cast<double>(conversion_factor),
                                 0, max_fractional_digits),
              base::FormatDouble(static_cast<double>(height_um) /
                                     static_cast<double>(conversion_factor),
                                 0, max_fractional_digits)),
          group};
}

}  // namespace

PaperWithSizeInfo::PaperWithSizeInfo(MediaSizeInfo msi,
                                     PrinterSemanticCapsAndDefaults::Paper p)
    : size_info(msi), paper(p) {}

MediaSizeInfo LocalizePaperDisplayName(const gfx::Size& size_um) {
  MediaSizeInfo size_info = InfoForStandardSize(size_um);
  return size_info.display_name.empty() ? InfoForUnregisteredSize(size_um)
                                        : size_info;
}

void SortPaperDisplayNames(std::vector<PaperWithSizeInfo>& papers) {
  std::vector<PaperWithSizeInfo> mm_sizes;
  std::vector<PaperWithSizeInfo> in_sizes;
  std::vector<PaperWithSizeInfo> named_sizes;

  // Break apart the list into separate sort groups.
  for (auto& p : papers) {
    switch (p.size_info.sort_group) {
      case MediaSizeGroup::kSizeMm:
        mm_sizes.emplace_back(p);
        break;

      case MediaSizeGroup::kSizeIn:
        in_sizes.emplace_back(p);
        break;

      case MediaSizeGroup::kSizeNamed:
        named_sizes.emplace_back(p);
        break;
    }
  }

  UErrorCode error = U_ZERO_ERROR;
  std::unique_ptr<icu::Collator> collator(icu::Collator::createInstance(error));
  DCHECK(U_SUCCESS(error));

  // Sort dimensional sizes (inch and mm) by width, then height.
  auto size_sort = [](const PaperWithSizeInfo& a, const PaperWithSizeInfo& b) {
    const gfx::Size& size_a = a.paper.size_um();
    const gfx::Size& size_b = b.paper.size_um();

    if (size_a.width() != size_b.width())
      return size_a.width() < size_b.width();
    return size_a.height() < size_b.height();
  };
  std::sort(mm_sizes.begin(), mm_sizes.end(), size_sort);
  std::sort(in_sizes.begin(), in_sizes.end(), size_sort);

  // Sort named sizes by name, then width, then height.
  auto name_sort = [&collator](const PaperWithSizeInfo& a,
                               const PaperWithSizeInfo& b) {
    const gfx::Size& size_a = a.paper.size_um();
    const gfx::Size& size_b = b.paper.size_um();

    UCollationResult comp = base::i18n::CompareString16WithCollator(
        *collator, a.size_info.display_name, b.size_info.display_name);
    if (comp != UCOL_EQUAL)
      return comp == UCOL_LESS;

    // Same name.  Sort by width, then height.
    if (size_a.width() != size_b.width())
      return size_a.width() < size_b.width();
    return size_a.height() < size_b.height();
  };
  std::sort(named_sizes.begin(), named_sizes.end(), name_sort);

  // Replace the original list with the newly sorted groups.
  papers.clear();
  papers.insert(papers.end(), in_sizes.begin(), in_sizes.end());
  papers.insert(papers.end(), mm_sizes.begin(), mm_sizes.end());
  papers.insert(papers.end(), named_sizes.begin(), named_sizes.end());
}

}  // namespace printing
